using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using CsGL.OpenGL;
using System.IO;

namespace CollDetect
{
	/// <summary>
	/// Custom OpenGL canvas control for the Coll Editor
	/// </summary>
	/// <remarks>
	/// <author>Xianhang Zhang</author>
	///	<date>27/5/2008</date>
	///	<revison>1.0</revison>
	///	</remarks>
    public class CollView : OpenGLControl
	{
        int[][] patchData;
        double[][] vertexData;
        public void loadModel(string filename)
        {
            using (StreamReader sr = new StreamReader(filename))
            {
                int numPatches = int.Parse(sr.ReadLine());
                patchData = new int[numPatches][];
                for (int i = 0; i < numPatches; i++)
                {
                    string[] patchStrings = sr.ReadLine().Split(',');
                    patchData[i] = new int[16];
                    for (int j = 0; j < 16; j++)
                        patchData[i][j] = int.Parse(patchStrings[j]) - 1;
                }
                int numVertices = int.Parse(sr.ReadLine());
                vertexData = new double[numVertices][];
                for (int i = 0; i < numVertices; i++)
                {
                    string[] vertexStrings = sr.ReadLine().Split(',');
                    vertexData[i] = new double[3];
                    vertexData[i][0] = double.Parse(vertexStrings[0]);
                    vertexData[i][1] = double.Parse(vertexStrings[2]) - 2.5;
                    vertexData[i][2] = double.Parse(vertexStrings[1]);
                }
            }
            for (int i = 0; i < patchData.Length; i++)
                storePatch(this.patchData[i]);
        }
        protected double basis(double t, int n)
        {
            switch (n)
            {
                case 0: return ((1 - t) * (1 - t) * (1 - t));
                case 1: return (3 * t * (1 - t) * (1 - t));
                case 2: return (3 * t * t * (1 - t));
                case 3: return (t * t * t);
            }
            throw new IndexOutOfRangeException("Invalide t value for basis function");
        }

        protected double[] curve(int[] patch, double u, double v, int resolution)
        {
            u /= resolution;
            v /= resolution;
            double x = 0, y = 0, z = 0;
            for (int i = 0; i < 4; i++)
                for (int j = 0; j < 4; j++)
                {
                    x += vertexData[patch[i * 4 + j]][0] * basis(u, i) * basis(v, j);
                    y += vertexData[patch[i * 4 + j]][1] * basis(u, i) * basis(v, j);
                    z += vertexData[patch[i * 4 + j]][2] * basis(u, i) * basis(v, j);
                }
            return new double[] { x, y, z };
        }

        NaiveCollide nc = new NaiveCollide();

        private void storePatch(int[] patch)
        {
            int resolution = 10;
            for (int i = 0; i < resolution; i++)
            {
                for (int j = 0; j < resolution; j++)
                {
                    Polygon p1 = new Polygon(curve(patch, i, j, resolution), curve(patch, i + 1, j, resolution), curve(patch, i, j + 1, resolution));
                    Polygon p2 = new Polygon(curve(patch, i, j + 1, resolution), curve(patch, i + 1, j, resolution), curve(patch, i + 1, j + 1, resolution));
                    nc.Add(p1);
                    nc.Add(p2);
                }
            }
        }
        /// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

        public CollView() : base()
		{
			// This call is required by the sizes.Forms Form Designer.
			InitializeComponent();
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if( components != null )
					components.Dispose();
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            //this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.CollView_MouseMove);
        }
		#endregion

        private void CollView_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
        {
        }


		protected override void OnPaint(PaintEventArgs pe)
		{
			base.OnPaint(pe);
		}

		protected override void OnResize(EventArgs e)
		{
			glResize();
			base.OnResize (e);
		}

        public float[] LightAmbient =  { 0.5f, 0.5f, 0.5f, 1.0f };
        public float[] LightDiffuse =  { 1.0f, 1.0f, 1.0f, 1.0f };
        public float[] LightPosition = { 0.0f, 3.0f, 2.0f, 1.0f };
        public float[] LightPosition2 = { 0.0f, -3.0f, -8.0f, 1.0f };
        
        protected override void InitGLContext() 
		{
            GL.glShadeModel(GL.GL_SMOOTH);									// Enable Smooth Shading
            GL.glClearColor(0.0f, 0.0f, 0.0f, 0.5f);						// Black Background
            GL.glClearDepth(1.0f);											// Depth Buffer Setup
            GL.glEnable(GL.GL_DEPTH_TEST);									// Enables Depth Testing
            GL.glDepthFunc(GL.GL_LEQUAL);									// The Type Of Depth Testing To Do
            GL.glHint(GL.GL_PERSPECTIVE_CORRECTION_HINT, GL.GL_NICEST);		// Really Nice Perspective Calculations

            GL.glLightfv(GL.GL_LIGHT1, GL.GL_AMBIENT, this.LightAmbient);	// Setup The Ambient Light
            GL.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, this.LightDiffuse);	// Setup The Diffuse Light
            GL.glLightfv(GL.GL_LIGHT1, GL.GL_POSITION, this.LightPosition);	// Position The Light
            //GL.glLightfv(GL.GL_LIGHT2, GL.GL_AMBIENT, this.LightAmbient);	// Setup The Ambient Light
            GL.glLightfv(GL.GL_LIGHT2, GL.GL_DIFFUSE, this.LightDiffuse);	// Setup The Diffuse Light
            GL.glLightfv(GL.GL_LIGHT2, GL.GL_POSITION, this.LightPosition2);	// Position The Light
            GL.glEnable(GL.GL_LIGHT1);										// Enable Light One
            GL.glEnable(GL.GL_LIGHT2);										// Enable Light One
            GL.glEnable(GL.GL_LIGHTING);
            // set material properties which will be assigned by glColor
            GL.glColorMaterial(GL.GL_FRONT, GL.GL_AMBIENT_AND_DIFFUSE);
            GL.glEnable(GL.GL_COLOR_MATERIAL);
            glResize();
		}
		
		protected void glResize()
		{
            GL.glClearColor(((float)this.BackColor.R / 255), ((float)this.BackColor.G / 255), ((float)this.BackColor.B / 255), 1.0f);
            GL.glMatrixMode(GL.GL_PROJECTION);
            GL.glLoadIdentity();
            GL.gluPerspective(45.0f, this.Size.Width / this.Size.Height, 0.1f, 100.0f);
            GL.gluLookAt(0, 0, 8, 0, 0, 0, 0, 1, 0);
            GL.glMatrixMode(GL.GL_MODELVIEW);
            GL.glLoadIdentity();
		}

        public bool CheckCollide()
        {
            return nc.isCollide(sphereLoc, sphereLoc);
        }

        public void look(double xAngle, double yAngle)
        {
            GL.glMatrixMode(GL.GL_PROJECTION);
            GL.glLoadIdentity();
            GL.gluPerspective(45.0f, this.Size.Width / this.Size.Height, 0.1f, 100.0f);
            GL.gluLookAt(8 * Math.Sin(xAngle), 0, 8 * Math.Cos(xAngle), 0, 0, 0, 0, 1, 0);
        }
        public double[] sphereLoc = new double[] { 1, 1, 0 };
        public override void glDraw()
		{
			GL.glClear( GL.GL_COLOR_BUFFER_BIT );
            GL.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);	// Clear the Screen and the Depth Buffer
            GL.glMatrixMode(GL.GL_MODELVIEW);		// Modelview Matrix
            GL.glLoadIdentity();					// reset the current modelview matrix
            GL.glTranslated(sphereLoc[0], sphereLoc[1], sphereLoc[2]);
            GL.glColor3f(0.0f, 1.0f, 0.0f);
            GL.glutSolidSphere(0.2, 10, 10);
            GL.glLoadIdentity();					// reset the current modelview matrix

            GL.glColor3f(0.5f, 0.5f, 0.5f);
            nc.Render();
        }

		public void repaint()
		{
			this.Invalidate();
		}
	}
}

